/** @file   lighttable.h
 * @brief   Declaration of LightTable - class.
 * @version $Revision: 1.3 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_LIGHTTABLE_H
#define H_WWW_LIGHTTABLE_H

#include "eng2d.h"
#include <vector>
#include <map>


namespace WeWantWar {

// Forward declarations
class LightSourceModel;


/** @class  LightSource
 * @brief   Represents a light source.
 * @author  Tomi Lamminsaari
 *
 * Every light source is an instace of this class.
 */
class LightSource
{
public:
  /** Constructor.
   * @param   aPosition         Position of this light source.
   * @param   aModelId          Id of the LightSourceModel this lightsource
   *                            uses.
   * @param   aIntensity        Intensity of this lightsource.
   */
  LightSource( const eng2d::Vec2D& aPosition, int aModelId, int aIntensity );
  
  /** Destructor.
   */
  virtual ~LightSource();
  
  /** Returns the id of the LightSourceModel this LightSource uses.
   * @return  The light model id.
   */
  int modelUid() const;
  
  /** Returns the position of this light source.
   * @return  Position of this light source.
   */
  eng2d::Vec2D position() const;
  
  /** Returns the intensity of this light source.
   * @return  Intensity.
   */
  int intensity() const;
  
protected:
  /** Id of the LightSourceModel this LightSource uses. */
  int           iModelUid;
  /** Position of this light source. */
  eng2d::Vec2D  iPosition;
  /** Intensity of this lightsource. */
  int           iIntensity;
};


/** @class  LightTable
 * @brief   Manages the lights.
 * @author  Tomi Lamminsaari
 *
 * LightTable allows you to add lightsources on the playfield.
 */
class LightTable
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** Constructor.
   */
  LightTable();
  
  /** Destructor.
   */
  virtual ~LightTable();
  
private:
  LightTable( const LightTable& aAnother );
  LightTable& operator = ( const LightTable& aAnother );
public:
  
  ///
  /// Methods
  /// =======
  
  /** Adds new lightsource. The default blender-function and default alpha-
   * value are being used.
   * @param   aLightSource      Pointer to LightSource object being added to
   *                            this table. Ownership is transferred. If null,
   *                            this method does nothing.
   */
  void add( LightSource* aLightSource );
  
  /** Registers the given LightSourceModel.
   * @param   aModel            Pointer to model object being registered.
   *                            Ownership is transferred.
   */
  void registerLightSourceModel( LightSourceModel* aModel );
  
    
  /** Removes all the lightsources from this light table. Should be called
   * at the end of the level.
   */
  void cleanup();
  
  /** Redraws the lights in lighttable.
   * @param     aCanvas           The bitmap where to render.
   * @param     aOffset           Map's TopLeft corner scroll offset.
   */
  void redraw( BITMAP* aCanvas, const eng2d::Vec2D& aOffset );

  /** Removes the the LightSource from given index.
   * @param   aIndex            Index of the LightSource being removed. If there
   *                            is no object in given index, this method does
   *                            nothing.
   */
  void removeLightSource( int aIndex );
  
  /** Removes the LightSourceModel that has the given uid.
   * @param   aModelUid         Uid of the model being removed.
   */
  void removeLightSourceModel( int aUid );


  ///
  /// Getter methods
  /// ==============
  
  /** Returns the LightSource from given index.
   * @param   aIndex            Index of the LightSource being returned.
   * @return  Pointer to LightSource. Ownership is not transferred.
   */
  LightSource* getLightSource( int aIndex ) const;
  
  /** Returns the LightSourceModel by its uid.
   * @param   aModelUid         Uid of the model being returned.
   * @return  Pointer to LightSourceModel that has the given uid or nullpointer
   *          if no matches were found.
   */
  LightSourceModel* findLightSourceModel( int aModelUid ) const;

private:

  ///
  /// Private members
  /// ===============
  
  /** The lightsources are stored in this vector. */
  std::vector<LightSource*> iLightsources;
  
  /** A datastructure where the registered LightSourceModel objects are
   * held.
   */
  std::map< int, LightSourceModel* > iLightModels;

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: lighttable.h,v $
 * Revision 1.3  2006/02/17 23:09:01  lamminsa
 * no message
 *
 * Revision 1.2  2006/02/11 23:41:03  lamminsa
 * New implementation. No more static methods.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:01+02  lamminsa
 * Initial revision
 *
 */
 
